<?php
/*
Plugin Name: AzuraCast Now Playing PRO
Plugin URI:  https://novoasema.net
Description: Tyylikäs AzuraCast-soitin WordPressiin. Näyttää nyt soivan kappaleen, kuuntelijat, LIVE DJ ja waveform-animaation.
Version:     1.0
Author:      Mika Ruohomäki
Author URI:  https://novoasema.net
License:     GPL2
*/

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

// ==============================
// AzuraCast Now Playing PRO Shortcode
// ==============================
function azuracast_nowplaying_pro_shortcode() {
    // === MUUTA NÄMÄ / chance this! ===
    $station_api = "https://cast.novoasema.net/api/nowplaying/novoasemanet";
    $stream_url  = "https://cast.novoasema.net/listen/novoasemanet/radio.mp3";
    // ===================

    ob_start();
    ?>

    <div id="azuracast-pro" class="azuracast-glass">
        <div class="ac-left">
            <div class="ac-cover">
                <img id="ac-cover-img" src="" alt="Album art" />
                <span id="ac-live-badge" class="ac-live">LIVE</span>
            </div>
        </div>

        <div class="ac-right">
            <div class="ac-top">
                <div class="ac-track">
                    <div id="ac-song" class="ac-song">Ladataan...</div>
                    <div id="ac-artist" class="ac-artist"></div>
                </div>
                <div class="ac-listeners" id="ac-listeners">—</div>
            </div>

            <!-- Waveform -->
            <div class="ac-wave-wrap" aria-hidden="true">
                <svg class="ac-wave" viewBox="0 0 600 60" preserveAspectRatio="none">
                    <path id="ac-wave-path" d="M0 30 C 50 10, 150 10, 200 30 C 250 50, 350 50, 400 30 C 450 10, 550 10, 600 30" fill="none" stroke="rgba(255,255,255,0.25)" stroke-width="2"/>
                </svg>
            </div>

            <!-- Progress + time -->
            <div class="ac-progress-row">
                <div id="ac-time-elapsed" class="ac-time">00:00</div>
                <div class="ac-progress-bar" id="ac-progress-bar" title="Progress">
                    <div id="ac-progress" class="ac-progress"></div>
                </div>
                <div id="ac-time-remaining" class="ac-time">-:--</div>
            </div>

            <!-- Controls -->
            <div class="ac-controls">
                <button id="ac-play" class="ac-btn">▶ Kuuntele</button>
                <div class="ac-volume">
                    <input id="ac-volume" type="range" min="0" max="1" step="0.01" value="1" />
                </div>
                <a id="ac-open" href="<?php echo esc_url($stream_url); ?>" target="_blank" rel="noopener" class="ac-link">Avaa stream</a>
            </div>

            <audio id="ac-audio" preload="auto" src="<?php echo esc_url($stream_url); ?>" muted></audio>
        </div>
    </div>

    <style>
    /* ======= Dark + Glassy Style ======= */
    #azuracast-pro { display:flex; gap:18px; align-items:center; max-width:760px; margin:12px 0; padding:16px; border-radius:14px; color:#fff; font-family: Inter, system-ui, -apple-system, "Segoe UI", Roboto, "Helvetica Neue", Arial; background: linear-gradient(180deg, rgba(20,20,25,0.75), rgba(12,12,16,0.6)); box-shadow: 0 8px 30px rgba(2,6,23,0.6); backdrop-filter: blur(8px) saturate(120%); -webkit-backdrop-filter: blur(8px) saturate(120%); border: 1px solid rgba(255,255,255,0.06); }
    #azuracast-pro .ac-left { flex: 0 0 auto; }
    #azuracast-pro .ac-cover { position:relative; width:140px; height:140px; border-radius:12px; overflow:hidden; box-shadow: 0 10px 30px rgba(0,0,0,0.6), inset 0 -10px 30px rgba(0,0,0,0.25); background: linear-gradient(180deg, rgba(255,255,255,0.02), rgba(0,0,0,0.12)); }
    #azuracast-pro .ac-cover img { width:100%; height:100%; object-fit:cover; display:block; transform: scale(1); transition: transform .5s ease; }
    #azuracast-pro:hover .ac-cover img { transform: scale(1.03); }
    #ac-live-badge { position:absolute; top:10px; left:10px; background: linear-gradient(90deg,#ff375f,#ff8a5a); color:white; font-weight:700; padding:6px 10px; border-radius:999px; font-size:12px; letter-spacing:0.6px; box-shadow: 0 4px 14px rgba(255,60,90,0.14); display:none; }

    #azuracast-pro .ac-right { flex:1 1 auto; min-width:0; }
    .ac-top { display:flex; justify-content:space-between; align-items:flex-start; gap:12px; }
    .ac-song { font-size:18px; font-weight:700; color:#fff; white-space:nowrap; overflow:hidden; text-overflow:ellipsis; max-width:60%; }
    .ac-artist { color: rgba(255,255,255,0.75); font-size:14px; margin-top:6px; }
    .ac-listeners { color:#9fb7ff; font-weight:600; font-size:13px; align-self:center; }

    .ac-wave-wrap { margin-top:12px; }
    .ac-wave { width:100%; height:40px; display:block; overflow:visible; }
    #ac-wave-path { stroke-width:2.5; stroke-linecap:round; stroke-linejoin:round; stroke: url(#gradWave); stroke-dasharray: 300; animation: waveMove 3.6s linear infinite; }
    @keyframes waveMove { 0%{stroke-dashoffset:0;opacity:.9} 50%{stroke-dashoffset:200;opacity:.7} 100%{stroke-dashoffset:0;opacity:.9} }

    .ac-progress-row { display:flex; align-items:center; gap:10px; margin-top:12px; }
    .ac-time { font-size:12px; color:rgba(255,255,255,0.7); min-width:44px; text-align:center; }
    .ac-progress-bar { flex:1; height:8px; background: rgba(255,255,255,0.06); border-radius:999px; overflow:hidden; position:relative; }
    .ac-progress { height:100%; width:0%; background: linear-gradient(90deg,#4facfe,#00f2fe); border-radius:999px; transition: width 0.5s linear; }

    .ac-controls { margin-top:14px; display:flex; align-items:center; gap:12px; flex-wrap:wrap; }
    .ac-btn { background: linear-gradient(180deg,#1dd1a1,#10ac84); border:none;color:#041014;padding:10px 14px;border-radius:10px;font-weight:700;cursor:pointer; box-shadow: 0 6px 22px rgba(10,160,120,0.12); transition: transform .15s ease; }
    .ac-btn:active { transform:scale(.98); }

    .ac-volume input[type="range"]{ -webkit-appearance:none; width:140px; height:6px; background:rgba(255,255,255,0.06); border-radius:999px; }
    .ac-volume input[type="range"]::-webkit-slider-thumb{ -webkit-appearance:none; width:14px; height:14px; border-radius:50%; background: #fff; box-shadow: 0 4px 12px rgba(0,0,0,0.35); border: 3px solid rgba(0,0,0,0.12); }

    .ac-link { color: #cfe4ff; text-decoration:none; background:transparent; font-weight:600; font-size:13px; padding:6px 10px; border-radius:8px; border:1px solid rgba(255,255,255,0.04); }

    @media (max-width:700px){ #azuracast-pro { flex-direction:column; align-items:center; text-align:center; padding:14px; } .ac-top { flex-direction:column; gap:6px; align-items:center; } .ac-song { max-width:100%; white-space:normal; } .ac-listeners { order:3; margin-top:6px; } .ac-controls { justify-content:center; } }
    </style>

    <svg style="position:absolute;width:0;height:0;overflow:hidden" aria-hidden="true">
        <defs>
            <linearGradient id="gradWave" x1="0%" y1="0%" x2="100%" y2="0%">
                <stop offset="0%" stop-color="#7ee8fa" stop-opacity="0.9"/>
                <stop offset="100%" stop-color="#4facfe" stop-opacity="0.9"/>
            </linearGradient>
        </defs>
    </svg>

    <script>
    (function(){
        const API_URL = "<?php echo esc_js($station_api); ?>";
        const audio   = document.getElementById("ac-audio");
        const playBtn = document.getElementById("ac-play");
        const volSlider = document.getElementById("ac-volume");
        const coverImg = document.getElementById("ac-cover-img");
        const songEl = document.getElementById("ac-song");
        const artistEl = document.getElementById("ac-artist");
        const listenersEl = document.getElementById("ac-listeners");
        const liveBadge = document.getElementById("ac-live-badge");
        const prog = document.getElementById("ac-progress");
        const progBar = document.getElementById("ac-progress-bar");
        const elapsedEl = document.getElementById("ac-time-elapsed");
        const remainingEl = document.getElementById("ac-time-remaining");

        window.addEventListener('load', () => {
            audio.muted = true; audio.volume = Number(volSlider.value || 1);
            audio.play().catch(()=>{});
        });

        const firstInteraction = () => { audio.muted=false; document.removeEventListener('click', firstInteraction); document.removeEventListener('keydown', firstInteraction); };
        document.addEventListener('click', firstInteraction);
        document.addEventListener('keydown', firstInteraction);

        playBtn.addEventListener('click', () => {
            if(audio.paused){ audio.play().then(()=>{playBtn.textContent="⏸ Pysäytä";}).catch(()=>{}); }
            else{ audio.pause(); playBtn.textContent="▶ Kuuntele"; }
        });

        volSlider.addEventListener('input',()=>{audio.volume=Number(volSlider.value);});

        const formatTime = sec => { if(isNaN(sec)||sec===null) return "00:00"; sec=Math.floor(Math.max(0,sec)); const m=Math.floor(sec/60); const s=sec%60; return String(m).padStart(2,'0')+":"+String(s).padStart(2,'0'); };

        async function updateNowPlaying() {
            try {
                const res = await fetch(API_URL,{cache:"no-store"});
                const data = await res.json();

                let song = (data.now_playing && (data.now_playing.song.text || (data.now_playing.song.artist+" - "+data.now_playing.song.title))) || "Ei tietoa";
                let artist = (data.now_playing && data.now_playing.song && data.now_playing.song.artist) || "";
                let art = (data.now_playing && data.now_playing.song && data.now_playing.song.art) || "";
                let listeners = (data.listeners && (data.listeners.current || data.listeners.total)) || (data.listeners || 0);
                let isLive = (data.now_playing && (data.now_playing.live || data.now_playing.is_stream_live || data.now_playing.streamer || data.now_playing.is_live)) || false;

                songEl.textContent=song;
                artistEl.textContent=artist;
                listenersEl.textContent="Kuuntelijoita nyt: "+listeners;
                coverImg.src = art || "";

                liveBadge.style.display = isLive?"inline-block":"none";

                const played = (data.now_playing && data.now_playing.played_seconds) || null;
                const duration = (data.now_playing && data.now_playing.song && data.now_playing.song.duration) || null;
                if(played!==null && duration!==null && duration>0){ prog.style.width=Math.min(1,played/duration)*100+"%"; elapsedEl.textContent=formatTime(played); remainingEl.textContent="-"+formatTime(duration-played); }
                else{ prog.style.width="0%"; elapsedEl.textContent="00:00"; remainingEl.textContent="-:--"; }

            } catch(err){
                songEl.textContent="Virhe haettaessa tietoja";
                artistEl.textContent=""; listenersEl.textContent="";
                console.error("AzuraCast API error:",err);
            }
        }

        updateNowPlaying();
        setInterval(updateNowPlaying,10000);

        setInterval(()=>{ const w=document.getElementById('ac-wave-path'); if(!w) return; w.style.opacity = audio && !audio.paused?1:0.45; w.style.transform="translateX(0)"; },600);

    })();
    </script>

    <?php
    return ob_get_clean();
}

add_shortcode('nowplaying_pro', 'azuracast_nowplaying_pro_shortcode');

